
/*
* ============================================================================
*  Name     : CShapeResolver from shaperesolver.h
*  Part of  : shaperesolver
*  Created  : 17/11/2003 by Forum Nokia
*  Description:
*     Impelementation of a custom resolver.
*  Version  : 1.0
*  Copyright: Forum Nokia
* ============================================================================
*/

#ifndef __SHAPERESOLVER_H__
#define __SHAPERESOLVER_H__

// INCLUDES
#include <ecom/resolver.h>


/**
 * Class:       CShapeResolver
 *
 * Description: An implementation of the CResolver definition. This controls
 *              the identification, (resolution), of which implementation
 *              will be used to satisfy an interface implementation
 *              instantiation.
 */
class CShapeResolver : public CResolver
    {
public:
    /**
     * Function:    NewL
     *
     * Description: Create instance of concrete implementation. The method
     *              takes a MPublicRegistry, allowing access to a list of
     *              implementations for a particular interface
     *
     * Param:       aRegistry provides a list of implementations of a specified
     *              interface.
     *
     * Returns:     Instance of this class.
     *
     */
    static CShapeResolver* NewL (MPublicRegistry& aRegistry);

    /**
     * Function:    ~CShapeResolver
     *
     * Description: Destroy the object
     */
    virtual ~CShapeResolver();

public: // CResolver
    /**
     * Function:    IdentifyImplementationL
     *
     * Description: Identifies the most appropriate implementation of a
     *              specified interface.
     *
     * Param:       aInterfaceUid is a unique id specific to definition
     *              interface.
     * Param:       aAdditionalParameters The parameters which must match for an
     *              implementation to be suitable.
     *
     * Return:      The UID of the implementation which satisfies the
     *              specified parameters. If there is no suitable
     *              implementation, then KNullUid is returned.
     */
    virtual TUid IdentifyImplementationL(TUid aInterfaceUid,
        const TEComResolverParams& aAdditionalParameters) const;

    /**
     * Function:    ListAllL
     *
     * Description: Lists all the implementations of the specified interface
     *              definition that satisfy the supplied resolution parameters.
     *
     * Param:       aInterfaceUid The UID of the interface of which to list
     *              implementations
     * Param:       aAdditionalParameters The parameters that must match for an
     *              implementation to be suitable
     *              implementation to be suitable
     *
     * Return:      Pointer to an array of suitable implementations. Ownership
     *              of this array is passed to the calling function, which
     *              should close and delete the array when its use is complete
     */
    virtual RImplInfoArray* ListAllL(TUid aInterfaceUid,
        const TEComResolverParams& aAdditionalParameters) const;

private:
    /**
     * Function:   CShapeResolver
     *
     * Discussion: Constructor for this object.
     *
     * Param:      aRegistry provides a list of implementations of a specified
     *             interface.
     */
    CShapeResolver(MPublicRegistry& aRegistry);

    /**
     * Function:    Resolve
     *
     * Description: Search algorithm implemented to find the requested
     *              interface implementation.
     *
     * Param:       aImplementationsInfo Information on the potential
     *              implementations
     * Param:       aAdditionalParameters The data to match against to detemine
     *              the implementation
     *
     * Return:      If found returns the Uid of the implementation, else it
     *              returns KNullUid
     */
    TUid Resolve(const RImplInfoArray& aImplementationsInfo,
        const TEComResolverParams& aAdditionalParameters) const;

    /**
     * Function:    IdentifyImplementationL
     *
     * Description: Method used to find a match between two implementation
     *              types
     *
     * Param:       aImplementationType The implementation data type to search
     *              for a match
     * Param:       aMatchType The data to search against
     * Param:       aUseWildcards ETrue if wildcard matching should be allowed
     *
     * Return:      ETrue is returned if found else EFalse
     */
    TBool Match(const TDesC8& aImplementationType, const TDesC8& aMatchType,
        TBool aUseWildcards) const;
private:
    mutable RImplInfoArray* iImplementationInfoArray;
    };

#endif

